import { TPoi, TNode, TCctv, TVms, TVds, TRse, TBit, TPark, TBit2, TStation, TBus, TIncident } from "./obj-facility.js";
import { TPolyline, _trafMap, _linkMap, _ifscMap, _roadMap } from "./obj-traffic.js";
import { LayerIndex, LayerType } from "./map-const.js";

// vworld :  6   7  8  9 10 11 12 13 14 15 16 17 18 19
// kakao  : 14  13 12 11 10  9  8  7  6  5  4  3  2  1
// naver  :  1   2  3  4  5  6  7  8  9 10 11 12 13 14 ==> 이건 확인해봐야함 API 마다 다름.

const MAX_ZOOM_LEVEL = 19;
export class TMapLayer {
    constructor(AMap, ALyrType, ALyrInfo, AImgW, AImgH, AVisible, AScales) {
        this.map = AMap;
        this.lyrInfo = ALyrInfo;
        this.lyrType = ALyrType;
        this.lyrVisible = AVisible;
        this.lyrIdx = ALyrInfo.index;
        this.lyrName = ALyrInfo.name;
        this.image = "/images" + ALyrInfo.image;
        this.showText = false;
        this.imgW = Number(AImgW);
        this.imgH = Number(AImgH);
        this.scales = new Array(MAX_ZOOM_LEVEL + 1);
        this.visible = new Array(MAX_ZOOM_LEVEL + 1);
        this.zoom = 0;
        this.layer = null;
        this.objMap = new Map();
        this.backPolyColor = "#000"; //'#FFFFFF';
        this.showTooltip = true;
        this.editMode = false;

        const scaleArr = AScales.split("^");
        for (let ii = 0; ii <= MAX_ZOOM_LEVEL; ii++) {
            this.scales[ii] = 1.0;
            this.visible[ii] = true;
        }
        let zoomIdx = 10;
        for (let ii in scaleArr) {
            this.scales[zoomIdx] = Number(scaleArr[ii]);
            zoomIdx++;
            if (zoomIdx > MAX_ZOOM_LEVEL) {
                break;
            }
        }
    } // constructor

    setShowText(AIsShow) {
        this.showText = AIsShow;
    }
    setEditMode(AIsEditMode) {
        this.editMode = AIsEditMode;
    }
    setLayerVisible(ALevels) {
        for (let ii = 0; ii <= MAX_ZOOM_LEVEL; ii++) {
            this.visible[ii] = false;
        }
        for (let ii = 0; ii < ALevels.length; ii++) {
            const level = Math.min(MAX_ZOOM_LEVEL, Number(ALevels[ii]));
            this.visible[level] = true;
        }
    }
    initObject() {
        this.objMap.clear();
        this.objMap = new Map();
    }

    initLayer() {
        if (this.layer == null) {
            return;
        }
        const source = this.layer.getSource();
        if (source != null) {
            const features = source.getFeatures();
            for (let ii = 0; ii < features.length; ii++) {
                source.removeFeature(features[ii]);
            }
            source.clear();
        }
    }

    getZoom() {
        return Math.round(this.map.getView().getZoom());
    }

    createLayer(AMapKey, AFeatures, ALyrVisible) {
        if (this.layer == null) {
            let lyrName = this.lyrName;
            if (this.lyrIdx >= LayerIndex.Link1 && this.lyrIdx <= LayerIndex.Road5) {
                let zoom = this.visible.reduce((pre, value, idx) => {
                    if (value == true) {
                        pre.push(idx);
                    }
                    return pre;
                }, []);
                const zoomLevel = zoom.join(", ");
                lyrName = `${this.lyrName} [${zoomLevel}]`;
            }
            this.layer = new ol.layer.Vector({
                title: lyrName,
            });
        }

        try {
            this.layer.setSource(
                new ol.source.Vector({
                    features: AFeatures,
                })
            );
        } catch (error) {
            console.error("createLayer: failed, ", this.lyrName, AMapKey, error);
            console.error("createLayer: failed, ", newLayer);
            console.error("createLayer: failed, ", AFeatures);
        }
        this.map.removeLayer(this.layer);
        this.layer.setVisible(ALyrVisible);
        this.layer.setZIndex(10 + this.lyrIdx);
        //this.map.addLayer(this.layer);

        AFeatures.length = 0;
        AFeatures = null;
        return this.layer;
    }

    makePoiLayer(AJsonData) {
        this.zoom = this.getZoom();

        this.initObject();
        this.initLayer();

        let features = new Array();
        switch (this.lyrIdx) {
            case LayerIndex.Node:
                AJsonData.forEach((el) => {
                    const obj = new TNode(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    //obj.setMarkerStyle();
                    //features.push(obj.marker);
                });
                return null; //break;
            case LayerIndex.Cctv:
                AJsonData.forEach((el) => {
                    const obj = new TCctv(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Vms:
                AJsonData.forEach((el) => {
                    const obj = new TVms(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Vds:
                AJsonData.forEach((el) => {
                    const obj = new TVds(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Rse:
                AJsonData.forEach((el) => {
                    const obj = new TRse(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Park:
                AJsonData.forEach((el) => {
                    const obj = new TPark(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Bit:
                AJsonData.forEach((el) => {
                    const obj = new TBit(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Station:
                AJsonData.forEach((el) => {
                    const obj = new TStation(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Bit2:
                AJsonData.forEach((el) => {
                    const obj = new TBit2(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Bus:
                AJsonData.forEach((el, idx) => {
                    const obj = new TBus(this, el, idx);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                    console.log(obj);
                });
                break;
            case LayerIndex.Incd:
                AJsonData.forEach((el) => {
                    const obj = new TIncident(this, el);
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                break;
            case LayerIndex.Select:
                let selectFeature = new ol.Feature({
                    type: "select",
                    idx: this.lyrIdx,
                    geometry: new ol.geom.Point(AJsonData),
                });

                selectFeature.setStyle(
                    new ol.style.Style({
                        image: new ol.style.Icon({
                            anchor: [0.5, 0.5],
                            opacity: 1,
                            size: [this.imgW, this.imgH],
                            scale: 1,
                            src: this.image + ".png",
                        }),
                    })
                );
                features = new Array();
                features.push(selectFeature);
                const selectLyr = this.createLayer(this.lyrIdx, features, this.lyrVisible);
                this.map.addLayer(selectLyr);
                return selectFeature;
            default:
                console.error(`makePoiLayer: Unknown layer type, ${this.lyrName}, ${this.lyrIdx}`);
                return null;
        }

        return this.createLayer(this.lyrIdx, features, this.lyrVisible);
    }

    makeTrafficLayer(ARefObjMap, AJsonData) {
        this.zoom = this.getZoom();

        this.initObject();
        this.initLayer();

        let features = new Array();
        AJsonData.forEach((el) => {
            const refObj = ARefObjMap.get(el.id);
            if (refObj) {
                const obj = new TPolyline(this, refObj, el);
                this.objMap.set(refObj.NMBR, obj);
                const marker = obj.getMarker(this.zoom);
                features.push(obj.marker);
            } else {
                console.warn(`makeTrafLayer: not found obj: ${this.lyrName}, ${el.id}`);
            }
        });

        const trafLayer = this.createLayer(this.lyrIdx, features, this.lyrVisible && this.visible[this.zoom]);
        if (this.editMode) {
            this.refreshEditLayer();
        } else {
            this.refreshLayer(false);
        }
        return trafLayer;
    }

    refreshLayer(AImmediate) {
        // 현재 숨김이거나 줌레벨이 숨김이면 모두 숨긴다.
        if (this.layer == null) {
            return;
        }
        if (this.lyrVisible == false) {
            if (this.layer.getVisible()) {
                this.layer.setVisible(false);
            }
            return;
        }
        if (this.editMode) {
            this.refreshEditLayer();
            return;
        }
        this.zoom = this.getZoom();
        const zoomVisible = this.visible[this.zoom];
        if (zoomVisible == false) {
            if (this.layer.getVisible()) {
                this.layer.setVisible(false);
            }
            return;
        }

        if (zoomVisible == false) {
            return;
        }

        this.objMap.forEach((obj) => {
            obj.setMarkerStyle(this.zoom, AImmediate);
        });
        if (!this.layer.getVisible()) {
            this.layer.setVisible(true);
        }
    }

    refreshEditLayer() {
        this.zoom = this.getZoom();
        this.objMap.forEach((obj) => {
            obj.setEditMarkerStyle(this.zoom);
        });
    }

    // 객체 맵 반환
    getLayerObject() {
        if (this.lyrIdx == LayerIndex.Traffic) {
            return _trafMap;
        } else if (this.lyrIdx >= LayerIndex.Link1 && this.lyrIdx <= LayerIndex.Link5) {
            return _linkMap;
        } else if (this.lyrIdx >= LayerIndex.Ifsc1 && this.lyrIdx <= LayerIndex.Ifsc5) {
            return _ifscMap;
        } else if (this.lyrIdx >= LayerIndex.Road1 && this.lyrIdx <= LayerIndex.Road5) {
            return _roadMap;
        }
        return this.objMap;
    }

    // 객체맵에서 객체를 찾아 리턴한다.
    findObject(ANmbr) {
        const obj = this.objMap.get(ANmbr);
        if (!obj) {
            console.error(`findObject: Unknown object, ${this.lyrIdx}, ${this.lyrName}, ${ANmbr}`);
        }
        return obj;
    }
    deleteFcltObject(ANmbr) {
        const obj = this.findObject(ANmbr);
        if (obj) {
            const source = this.layer.getSource();
            if (source != null) {
                source.removeFeature(obj.marker);
            }
            this.objMap.delete(obj.NMBR);
        }
    }
    initFcltObject(AFcltObjs) {
        this.zoom = this.getZoom();

        this.initObject();
        this.initLayer();

        let features = new Array();
        switch (this.lyrIdx) {
            case LayerIndex.Cctv:
            case LayerIndex.Vms:
            case LayerIndex.Vds:
            case LayerIndex.Rse:
            case LayerIndex.Park:
            case LayerIndex.Bit:
                AFcltObjs.forEach((el) => {
                    const obj = new TPoi(this, el.NMBR, el.NMBR, el.NAME, el.LAT, el.LNG, el.COMM, 0, "");
                    this.objMap.set(obj.NMBR, obj);
                    obj.setMarkerStyle(this.zoom);
                    features.push(obj.marker);
                });
                //console.log(this);
                return this.createLayer(this.lyrIdx, features, this.lyrVisible);
            //break;
            default:
                console.error(`initFcltObject: Unknown layer type, ${this.lyrName}, ${this.lyrIdx}`);
                break;
        }
        return null;
    }

    // 모든 객체를 삭제한다.
    removeLayerObject() {
        this.initObject();
        this.initLayer();
    }
    addFeature(AFeature) {
        this.layer.getSource().addFeature(AFeature);
    }

    createFcltObject(ANmbr, AId, AName, AComm, AXCrdn, AYCrdn) {
        if (!this.layer || !this.layer.getSource()) {
            console.error(`createFcltObject: layer or feature source null, ${this.lyrName}, ${this.lyrIdx}`);
            return null;
        }
        this.zoom = this.getZoom();

        switch (this.lyrIdx) {
            case LayerIndex.Node:
                return null;
            case LayerIndex.Cctv:
            case LayerIndex.Vms:
            case LayerIndex.Vds:
            case LayerIndex.Rse:
            case LayerIndex.Park:
            case LayerIndex.Bit:
                const obj = new TPoi(this, ANmbr, AId, AName, AXCrdn, AYCrdn, AComm, "0", 0, "");
                this.objMap.set(obj.NMBR, obj);
                obj.setMarkerStyle(this.zoom);
                obj.edit = true;
                this.layer.getSource().addFeature(obj.marker);
                return obj;
            default:
                console.error(`createFcltObject: Unknown layer type: ${this.lyrName}, ${this.lyrIdx}`);
                return null;
        }
    }

    // 레이터 퓨처 전체영역으로 확대
    extentLayer() {
        if (this.layer == null) {
            return;
        }
        const source = this.layer.getSource();
        if (source != null) {
            const features = source.getFeatures();
            if (features.length > 0) {
                const extent = source.getExtent();
                this.map.getView().fit(extent, this.map.getSize());
            }
        }
    }
}
